# Profile & Staff Management Guide

## Overview
Two new sections have been added to the admin dashboard:
1. **Profile Management** - Edit admin profile details and change password
2. **Staff Management** - Manage cleaning staff members and assign services

---

## 1. PROFILE SECTION

### Features
- **View Profile**: Display current admin profile information
- **Edit Profile**: Update name, phone number, and details/bio
- **Change Password**: Securely change admin password with confirmation

### How to Use

#### Accessing Profile
1. Click on "Profile" in the left sidebar
2. View your current profile information
3. Click "Edit Profile" to make changes

#### Editing Profile
1. Click "Edit Profile" button
2. Update the following fields:
   - **Full Name**: Your name
   - **Phone Number**: Contact phone
   - **Details/Bio**: Add your professional details
   - **New Password** (optional): Leave blank to keep current password
   - **Confirm Password**: Must match the new password

3. Click "Save Changes"

### Database Schema
The `users` table has been updated with a `details` column:
```sql
ALTER TABLE users ADD COLUMN details TEXT;
```

### API Endpoints

#### Get Admin Profile
```
GET api_handler.php?action=get_admin_profile
```
Returns:
```json
{
  "success": true,
  "data": {
    "id": 1,
    "name": "Admin Name",
    "email": "admin@example.com",
    "phone": "1234567890",
    "details": "Profile details"
  }
}
```

#### Update Admin Profile
```
POST api_handler.php
action: update_admin_profile
name: "New Name"
phone: "1234567890"
details: "Updated details"
password: "newpassword" (optional)
```

---

## 2. STAFF MANAGEMENT SECTION

### Features
- **Add Staff Members**: Create new staff profiles
- **Edit Staff Members**: Update staff information and assigned services
- **Delete Staff Members**: Remove staff from the system
- **Assign Services**: Link staff members to specific services
- **View All Staff**: See all staff members with their details
- **Real-time Integration**: All changes update instantly

### How to Use

#### Adding Staff Member
1. Click "Staff Management" in sidebar
2. Click "Add Staff Member" button
3. Fill in the following fields:
   - **Name**: Staff member's full name (required)
   - **Email**: Unique email address (required)
   - **Phone**: Contact number (required)
   - **Position/Role**: e.g., "Cleaner", "Supervisor" (required)
   - **Assigned Services**: Select one or more services
   - **Status**: Active or Inactive

4. Click "Save Staff Member"

#### Editing Staff Member
1. Click "Staff Management" in sidebar
2. Find the staff member in the list
3. Click "Edit" button
4. Modify the information as needed
5. Click "Save Staff Member"

#### Deleting Staff Member
1. Click "Staff Management" in sidebar
2. Find the staff member in the list
3. Click "Delete" button
4. Confirm the deletion

#### Assigning Services to Staff
1. When adding or editing staff
2. In the "Assigned Services" dropdown:
   - Hold **Ctrl** (Windows) or **Cmd** (Mac) to select multiple services
   - Click on each service to select/deselect
3. Click "Save Staff Member"

### Database Schema

#### Staff Table
```sql
CREATE TABLE staff (
  id INT PRIMARY KEY AUTO_INCREMENT,
  name VARCHAR(255) NOT NULL,
  email VARCHAR(255) UNIQUE NOT NULL,
  phone VARCHAR(20) NOT NULL,
  position VARCHAR(100) NOT NULL,
  status ENUM('active', 'inactive') DEFAULT 'active',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
)
```

#### Staff Services Mapping Table
```sql
CREATE TABLE staff_services (
  id INT PRIMARY KEY AUTO_INCREMENT,
  staff_id INT NOT NULL,
  service_id INT NOT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (staff_id) REFERENCES staff(id) ON DELETE CASCADE,
  FOREIGN KEY (service_id) REFERENCES services(id) ON DELETE CASCADE,
  UNIQUE KEY unique_staff_service (staff_id, service_id)
)
```

### API Endpoints

#### Get All Staff
```
GET api_handler.php?action=get_staff
```
Returns:
```json
{
  "success": true,
  "data": [
    {
      "id": 1,
      "name": "John Doe",
      "position": "Cleaner",
      "email": "john@example.com",
      "phone": "1234567890",
      "status": "active",
      "services_count": 3
    }
  ]
}
```

#### Get Staff Details
```
GET api_handler.php?action=get_staff_details&id=1
```
Returns:
```json
{
  "success": true,
  "data": {
    "id": 1,
    "name": "John Doe",
    "email": "john@example.com",
    "phone": "1234567890",
    "position": "Cleaner",
    "status": "active",
    "services": [
      {"id": 1},
      {"id": 2}
    ]
  }
}
```

#### Add Staff
```
POST api_handler.php
action: add_staff
name: "Staff Name"
email: "staff@example.com"
phone: "1234567890"
position: "Cleaner"
status: "active"
services: ["1", "2", "3"]
```

#### Edit Staff
```
POST api_handler.php
action: edit_staff
staff_id: 1
name: "Updated Name"
email: "updated@example.com"
phone: "1234567890"
position: "Supervisor"
status: "active"
services: ["1", "2"]
```

#### Delete Staff
```
POST api_handler.php
action: delete_staff
staff_id: 1
```

---

## Installation Instructions

### 1. Update Database
Run the database setup script to create the new tables:
```
1. Navigate to: http://localhost/Cleaning%20Service/setup_database.php
2. The script will create staff and staff_services tables
3. Refresh the page to confirm success
```

Or manually add columns if using existing database:
```sql
-- Add details column to users table if not exists
ALTER TABLE users ADD COLUMN details TEXT;

-- Create staff table
CREATE TABLE IF NOT EXISTS staff (
  id INT PRIMARY KEY AUTO_INCREMENT,
  name VARCHAR(255) NOT NULL,
  email VARCHAR(255) UNIQUE NOT NULL,
  phone VARCHAR(20) NOT NULL,
  position VARCHAR(100) NOT NULL,
  status ENUM('active', 'inactive') DEFAULT 'active',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);

-- Create staff_services mapping table
CREATE TABLE IF NOT EXISTS staff_services (
  id INT PRIMARY KEY AUTO_INCREMENT,
  staff_id INT NOT NULL,
  service_id INT NOT NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (staff_id) REFERENCES staff(id) ON DELETE CASCADE,
  FOREIGN KEY (service_id) REFERENCES services(id) ON DELETE CASCADE,
  UNIQUE KEY unique_staff_service (staff_id, service_id)
);
```

### 2. Updated Files
The following files have been updated:

#### `admin-dashboard.php`
- Added "Profile" menu item to sidebar
- Added "Staff Management" menu item to sidebar
- Added Profile section tab with edit button
- Added Staff Management section tab
- Added Profile Modal for editing admin details
- Added Staff Modal for adding/editing staff

#### `admin-dashboard.js`
- Added `loadProfile()` function to fetch profile data
- Added `renderProfileCard()` to display profile information
- Added `openEditProfileModal()` and `closeProfileModal()` for profile modal
- Added `loadStaff()` to fetch all staff members
- Added `renderStaffTable()` to display staff in table format
- Added `editStaff()` to load staff details for editing
- Added `deleteStaff()` to remove staff members
- Added `openAddStaffModal()` and `closeStaffModal()` for staff modal
- Added `loadServicesForStaff()` to populate service dropdowns
- Added form event listeners for profile and staff forms
- Updated tab titles object to include new sections

#### `api_handler.php`
- Added `get_admin_profile()` function
- Added `update_admin_profile()` function
- Added `getStaff()` function
- Added `getStaffDetails()` function
- Added `addStaff()` function
- Added `editStaff()` function
- Added `deleteStaff()` function
- Added case statements in switch for all new actions

#### `setup_database.php`
- Added `details` column to users table
- Added `staff` table creation
- Added `staff_services` table creation
- Updated completion message

---

## Features Implemented

### Profile Management
✅ View admin profile  
✅ Edit name  
✅ Edit phone number  
✅ Edit details/bio  
✅ Change password with confirmation  
✅ Password validation  
✅ Real-time updates  

### Staff Management
✅ Add new staff members  
✅ Edit existing staff information  
✅ Delete staff members  
✅ Assign multiple services to staff  
✅ View all staff members  
✅ See service count for each staff  
✅ Filter staff by status (Active/Inactive)  
✅ Real-time integration with API  
✅ Unique email validation  
✅ Responsive design for all devices  

---

## Security Notes

1. **Password Hashing**: Passwords are hashed using `password_hash()` with DEFAULT algorithm (bcrypt)
2. **Email Validation**: Emails are checked for uniqueness in the database
3. **Input Sanitization**: All inputs are escaped using `real_escape_string()`
4. **Authentication Check**: Only admins can access these sections
5. **Session Validation**: All operations require active admin session

---

## Responsive Design

Both sections are fully responsive:
- **Desktop (768px+)**: Full layout with side-by-side forms
- **Tablet (600px-768px)**: Stacked layout with optimized spacing
- **Mobile (<600px)**: Full-width forms, touch-friendly buttons

---

## Troubleshooting

### Staff Table Not Found
**Solution**: Run the setup_database.php file to create the required tables

### Services Not Showing in Dropdown
**Solution**: Ensure services exist in the services table. Add them through the Services Management section.

### Changes Not Saving
**Solution**: Check browser console for errors. Ensure api_handler.php is accessible and PHP is properly configured.

### Email Already Exists Error
**Solution**: The email is already used by another staff member. Use a unique email address.

---

## Future Enhancements

Possible improvements:
- Staff scheduling and availability management
- Performance ratings and reviews
- Work history and completed services
- Staff document verification (certifications, etc.)
- Staff availability calendar
- Customer-staff ratings system
- Assignment of staff to bookings

---

## Support

For issues or questions:
1. Check the troubleshooting section above
2. Verify database tables are created correctly
3. Check browser console for JavaScript errors
4. Review server logs for PHP errors
